import { DerEncodedPublicKey, KeyPair, Signature } from '@dfinity/agent';
import { PublicKey, SignIdentity } from '@dfinity/agent';
declare type PublicKeyHex = string;
declare type SecretKeyHex = string;
export declare type JsonableSecp256k1Identity = [PublicKeyHex, SecretKeyHex];
export declare class Secp256k1PublicKey implements PublicKey {
    /**
     * Construct Secp256k1PublicKey from an existing PublicKey
     * @param {PublicKey} key
     * @returns {Secp256k1PublicKey} Instance of Secp256k1PublicKey
     */
    static from(key: PublicKey): Secp256k1PublicKey;
    static fromRaw(rawKey: ArrayBuffer): Secp256k1PublicKey;
    static fromDer(derKey: DerEncodedPublicKey): Secp256k1PublicKey;
    private static derEncode;
    private static derDecode;
    private readonly rawKey;
    private readonly derKey;
    private constructor();
    toDer(): DerEncodedPublicKey;
    toRaw(): ArrayBuffer;
}
export declare class Secp256k1KeyIdentity extends SignIdentity {
    protected _privateKey: ArrayBuffer;
    /**
     * Generates an identity. If a seed is provided, the keys are generated from the
     * seed according to BIP 0032. Otherwise, the key pair is randomly generated.
     * This method throws an error in case the seed is not 32 bytes long or invalid
     * for use as a private key.
     * @param {Uint8Array} seed the optional seed
     * @returns {Secp256k1KeyIdentity}
     */
    static generate(seed?: Uint8Array): Secp256k1KeyIdentity;
    static fromParsedJson(obj: JsonableSecp256k1Identity): Secp256k1KeyIdentity;
    static fromJSON(json: string): Secp256k1KeyIdentity;
    /**
     * generates an identity from a public and private key. Please ensure that you are generating these keys securely and protect the user's private key
     * @param {ArrayBuffer} publicKey
     * @param {ArrayBuffer} privateKey
     * @returns {Secp256k1KeyIdentity}
     */
    static fromKeyPair(publicKey: ArrayBuffer, privateKey: ArrayBuffer): Secp256k1KeyIdentity;
    /**
     * generates an identity from an existing secret key, and is the correct method to generate an identity from a seed phrase. Please ensure you protect the user's private key.
     * @param {ArrayBuffer} secretKey
     * @returns {Secp256k1KeyIdentity}
     */
    static fromSecretKey(secretKey: ArrayBuffer): Secp256k1KeyIdentity;
    protected _publicKey: Secp256k1PublicKey;
    protected constructor(publicKey: Secp256k1PublicKey, _privateKey: ArrayBuffer);
    /**
     * Serialize this key to JSON-serializable object.
     * @returns {JsonableSecp256k1Identity}
     */
    toJSON(): JsonableSecp256k1Identity;
    /**
     * Return a copy of the key pair.
     * @returns {KeyPair}
     */
    getKeyPair(): KeyPair;
    /**
     * Return the public key.
     * @returns {Secp256k1PublicKey}
     */
    getPublicKey(): Secp256k1PublicKey;
    /**
     * Signs a blob of data, with this identity's private key.
     * @param {ArrayBuffer} challenge - challenge to sign with this identity's secretKey, producing a signature
     * @returns {Promise<Signature>} signature
     */
    sign(challenge: ArrayBuffer): Promise<Signature>;
}
export default Secp256k1KeyIdentity;
