import { PublicKey } from '@dfinity/agent';
import { DelegationChain } from '@dfinity/identity';
import { Principal } from '@dfinity/principal';
/**
 * Options for {@link createAuthenticationRequestUrl}. All these options may be limited
 * further by the identity provider, or an error can happen.
 */
export interface CreateUrlOptions {
    /**
     * The public key to delegate to. This should be the public key of the session key.
     */
    publicKey: PublicKey;
    /**
     * The scope of the delegation. This must contain at least one key and a maximum
     * of four. This is validated in {@link createAuthenticationRequestUrl} but also
     * will be validated as part of the identity provider.
     */
    scope: Array<string | Principal>;
    /**
     * The URI to redirect to, after authentication. By default, `window.location.origin`.
     */
    redirectUri?: string;
    /**
     * The URL base to use for the identity provider.
     * By default, this is "https://auth.ic0.app/authorize".
     */
    identityProvider?: URL | string;
}
/**
 * List of things to check for a delegation chain validity.
 */
export interface DelegationValidChecks {
    /**
     * Check that the scope is amongst the scopes that this delegation has access to.
     */
    scope?: Principal | string | Array<Principal | string>;
}
/**
 * A parsed access token.
 */
export declare type AccessToken = string & {
    _BRAND: 'access_token';
};
/**
 * Create a URL that can be used to redirect the browser to request authentication (e.g. using
 * the authentication provider). Will throw if some options are invalid.
 * @param options An option with all options for the authentication request.
 */
export declare function createAuthenticationRequestUrl(options: CreateUrlOptions): URL;
/**
 * Returns an AccessToken from the Window object. This cannot be used in Node, instead use
 * the {@link getAccessTokenFromURL} function.
 *
 * An access token is needed to create a DelegationChain object.
 */
export declare function getAccessTokenFromWindow(): AccessToken | null;
/**
 * Analyze a URL and try to extract an AccessToken from it.
 * @param url The URL to look into.
 */
export declare function getAccessTokenFromURL(url: URL | string): AccessToken | null;
/**
 * Create a DelegationChain from an AccessToken extracted from a redirect URL.
 * @param accessToken The access token extracted from a redirect URL.
 */
export declare function createDelegationChainFromAccessToken(accessToken: AccessToken): DelegationChain;
/**
 * Analyze a DelegationChain and validate that it's valid, ie. not expired and apply to the
 * scope.
 * @param chain The chain to validate.
 * @param checks Various checks to validate on the chain.
 */
export declare function isDelegationValid(chain: DelegationChain, checks?: DelegationValidChecks): boolean;
